from enum import IntFlag
from typing import Dict, List
from json import dumps

from patterns import *


class Parameter(metaclass=SingletonMeta):
    class OptimizeComponent(IntFlag):
        Nessuno = 0
        '''Il solver viene eseguito solo per verificare che esista una sol accettabbile.\n
        NON TIENE CONTO DI SODDISFARE I VINCOLI DI PREFERENZA DI NESSUNO, nemmeno di avere la somma delle penalità sotto una certa soglia'''
        Docenti = 1
        Orientamento = 2
        Studenti = 4
        
    
    def __init__(self):
        '''Crea un oggetto statico, accessibile da tutti i componenti con tutti i parametri configurabili del solver'''
        # super().__init__()
        
        self.folderInsegnamentiXml:List[str] = list()
        '''Lista di folders con i file xml degli Insegnamenti'''
        self.fileXmlAule:str = ""
        '''File xml contenente le Aule allocabili'''
        self.jsonFileDocenti:str = ""
        '''File json con i vari vincoli riguardanti i Docenti'''
        self.jsonFileInsegnamenti:str = ""
        '''File json con i vari vincoli riguardanti gli Insegnamenti'''
        self.nomeSolutions:str = ""
        '''Basename delle soluzioni generate dall'exec corrente'''
        self.timeLimitWork:int = -1
        '''Il tempo limite di exec per CPLEX. -1 significa no limit'''
        self.timeLimitTraSolutions:int = 60*60
        '''Se la funzione obiettivo non è migliorata di molto ma tra la precedente soluzione trovata e questa è passato almeno questo tempo 
        -> salvo anche la soluzione corrente'''
        self.nFilesInsegnamenti:int = 10000
        '''SOLO PER DEBUGGING: setta il limite di file da caricare (in teoria non ho 10k file, basta alzare il limite nel caso)'''
        self.modelConfig:Parameter.OptimizeComponent = Parameter.OptimizeComponent.Nessuno
        '''Lista dei componenti da ottimizzare'''
        self.additionalModelConfig:Parameter.OptimizeComponent = Parameter.OptimizeComponent.Nessuno
        '''Attiva i limiti addizionali sui vari componenti. Eg: la somma delle penalità di un Docente è < di un valore, la penalità media di 
        un docente è minore di un certo valore, ...'''
        self.sogliaInterestingSol:float = 0.2
        '''Indica il rapporto di costo della funzione obiettivo tra la prima sol trovata e la corrente. Le sol trovate vengono ritenute di 
        interesse solo una volta che siamo sotto tale soglia'''
        self.CPLEX_logging:bool = False
        self.CPLEX_LogPeriod:int = 500000
        '''Da modificare solo se voglio avere il logging di CPLEX su file (mettere 5k/10k)'''
        self.debuggingPenalty:bool = False
        '''Serve per avere un riscontro sulle penalità violate anche nel caso in cui stia solamente provando a verificare la validità di
        una soluzione. Cioè sto solo verificando se non esistono problemi tra gli hard constraint, ma sono anche interessato a vedere
        quali penalità sono scattate per la soluzione corrente'''
        self.nWorkers:int = -1
        '''Setta il numero di thread in esecuzione parallela. Di default corrisponde al numero di core della macchina'''
        
        self.usePianoAllocazioneAsBase:bool = False
        '''Stabilisce se la modalità di utilizzo risulta essere quella in cui buona parte degli Insegnamenti risultano essere già allocati con
        un Piano Allocazione precedente come hard constraint (tutti quelli non presenti della lista di ID_INC di Insegnamenti da allocare)'''
        self.listID_INCToModify:List[int] = list()
        '''Contiene tutti gli ID_INC degli Insegnamenti da modificare (quelli non presenti sono quelli preallocati in funzione del Piano 
        Allocazione già creato). In caso nelle folder xml ci siano anche file riguardanti Insegnamenti da mantenere preallocati, questi
        semplicemente verranno ignorati (l'allocazione del db fa fede)'''
        self.pianoAllocazioneBase:str = ""
        '''E' il nome del Piano Allocazione da usare come base. Se venisse specificato un nome invalido verrà notificato dai log ma potrebbe
        comunque partire la generazione di un Piano Allocazione non valido'''
        
        self.capienzaAuleExtended:int = 1
        '''Stabilisce di quanti livelli posso espandere la ricerca verso l'alto per soffisfare l'occupazione di uno Slot in termini di capienza:
        eg: uno Slot per default assegnabile ad un Aula di capienza Piccola può essere assegnato a:\n
            - capienzaAuleExtended = 0: solo un'Aula di capienza Piccola
            - capienzaAuleExtended = 1: un'Aula di capiena Piccola o MedioPiccola
            - ...
            - capienzaAuleExtended = 4: una qualsiasi Aula (ho 5 livelli di capienza delle Aule)'''
            
        self.sabatoEnabled:bool = False
        '''Abilita l'allocazione di Slot anche al Sabato'''
        self.nSlotSabato:int = 4
        '''Stabilisce il numero di Slot allocabili il Sabato (il max giornaliero è 7)'''

    
    def get_description(self) -> str:
        '''Genera una descrizione per la configurazione corrente'''
        desc = vars(self)
        return dumps(desc)
        
    
    def add_folderInsegnamentiXml(self, folderPath:str):
        '''Aggiunge una cartella contenente i file xml degli Insegnamenti (TemplateOrario)'''
        self.folderInsegnamentiXml.append(folderPath)
    
    def set_fileXmlLocaliAllocati(self, filePath:str):
        '''Setta il file xml contenente i Locali allocati per definire l'Orario'''
        self.fileXmlAule = filePath
        
    def set_jsonFileDocenti(self, filePath:str):
        '''Setta il file json contenente i vari constraint (hard e soft) definibili per i Docenti'''
        self.jsonFileDocenti = filePath
        
    def set_jsonFileInsegnamenti(self, filePath:str):
        '''Setta il file json contenente i vari constraint (hard e soft) definibili per stabilire le fasce in cui allocare/non allocare
        certi Insegnamenti'''
        self.jsonFileInsegnamenti = filePath
        
    def set_nomeSolutions(self, nomeSolutions:str):
        '''Setta il nome per le soluzioni trovate nell'exec corrente'''
        self.nomeSolutions = nomeSolutions
        
    def set_timeLimitWork(self, timeLimit = -1):
        '''Setta il tempo massimo di exec (in secondi)'''
        self.timeLimitWork = timeLimit
        
    def add_optimizeComponent(self, component:OptimizeComponent):
        '''Aggiunge un OptimizeComponenent alla lista dei tipi di soft constraint da ottimizzare'''
        self.modelConfig |= component
        
    def add_additionalModelConfig(self, component:OptimizeComponent):
        '''Aggiunge un OptimizeComponent addizionale (vedi descrizione variabile classe per maggiori dettagli)'''
        self.additionalModelConfig |= component
        
    def set_timeLimitTraSolutions(self, timeLimit:int):
        '''Setta il tempo limite tra due soluzioni (in secondi)'''
        self.timeLimitTraSolutions = timeLimit
        
    def set_sogliaInterestingSol(self, soglia:float):
        '''Setta la soglia minima per iniziare a salvare le sol'''
        self.sogliaInterestingSol = soglia
        
    def set_CPLEX_LogPeriod(self, logPeriod:int):
        '''Per limitare l'output di CPLEX in modo tale da averlo anche su file'''
        self.CPLEX_LogPeriod = logPeriod
        
    def set_CPLEXlogging(self, value:bool = False):
        '''Per abilitare/disabilitare il logging su file di CPLEX'''
        self.CPLEX_logging = value
        
    def set_usePianoAllocazioneAsBase(self, value:bool = False):
        '''Per abilitare/disabilitare questa particolare modalità di exec'''
        self.usePianoAllocazioneAsBase = value
        
    def add_listID_INCToModify(self, listOfID_INC:List[int]):
        '''Estende la lista di ID_INC da non precaricare dal Piano Allocazione.\n
        NOTE: gli insegnamenti in questa lista sono gli unici caricati dai file xml'''
        self.listID_INCToModify.extend(listOfID_INC)
        
    def set_pianoAllocazioneBase(self, pianoAllocazione:str):
        '''Setta il Piano Allocazione base da cui partire per preallocare buona parte degli Insegnamenti'''
        self.pianoAllocazioneBase = pianoAllocazione
        
    def set_debuggingPenalty(self, value:bool = False):
        '''Creo nel modello tutte le variabili che tengono traccia delle penalità anche se non sono interessato a minimizzarle'''
        self.debuggingPenalty = value
        
    def set_nWorkers(self, nWorkers:int):
        '''Setta il numero di thread di CPLEX'''
        self.nWorkers = nWorkers
        
    def set_capienzaAuleExtended(self, levelExtended:int):
        '''Setta il livello di estenzione di capienza per allocare gli Slot nelle Aule (vedi self.capienzaAuleExtended per ulteriori info)'''
        self.capienzaAuleExtended = levelExtended
        
    def set_sabatoEnabled(self, sabatoEnabled:bool):
        '''Abilita/disabilita il Sabato'''
        self.sabatoEnabled = sabatoEnabled
        
    def set_nSlotSabato(self, nSlotSabato:int):
        '''Setta il numero di Slot allocabili il Sabato. 4 di default'''
        self.nSlotSabato = nSlotSabato